/*
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "sig_ide_bus.h"

void
sig_ide_bus_reset(struct sig_ide_bus *b, unsigned int val)
{
	unsigned int nr;

	for (nr = 0; nr < b->nmembers; nr++) {
		void (*func)(void *, unsigned int);

		func = b->member[nr].funcs->reset;
		if (func) {
			func(b->member[nr].s, val);
		}
	}
}

void
sig_ide_bus_inw(struct sig_ide_bus *b, uint16_t port, uint16_t *valp)
{
	unsigned int nr;

	assert((/* 0 <= port && */ port <= 8)
	    || (8 + 6 <= port && port <= 8 + 7));

	for (nr = 0; ; nr++) {
		int (*func)(void *, uint16_t, uint16_t *);

		if (nr == b->nmembers) {
			/* Host has pull-down resistor on bit 7. */
			*valp = (uint16_t) ~(1 << 7);
			break;
		}
		func = b->member[nr].funcs->inw;
		if (func
		 && func(b->member[nr].s, port, valp) == 0) {
			break;
		}
	}
}

void
sig_ide_bus_outw(struct sig_ide_bus *b, uint16_t port, uint16_t val)
{
	unsigned int nr;

	assert((/* 0 <= port && */ port <= 8)
	    || port == 8 + 6);

	for (nr = 0; ; nr++) {
		void (*func)(void *, uint16_t, uint16_t);

		if (nr == b->nmembers) {
			return;
		}

		func = b->member[nr].funcs->outw;
		if (func) {
			func(b->member[nr].s, port, val);
		}
	}
}

void
sig_ide_bus_dmainw(struct sig_ide_bus *b, uint16_t *valp)
{
	unsigned int nr;
	int (*func)(void *, uint16_t *);

	for (nr = 0; ; nr++) {
		if (nr == b->nmembers) {
			/* Host has pull-down resistor on bit 7. */
			*valp = (uint16_t) ~(1 << 7);
			break;
		}
		func = b->member[nr].funcs->dmainw;
		if (func
		 && func(b->member[nr].s, valp) == 0) {
			break;
		}
	}
}

void
sig_ide_bus_dmaoutw(struct sig_ide_bus *b, uint16_t val)
{
	unsigned int nr;

	for (nr = 0; nr < b->nmembers; nr++) {
		void (*func)(void *, uint16_t);

		if (nr == b->nmembers) {
			return;
		}

		func = b->member[nr].funcs->dmaoutw;
		if (func) {
			func(b->member[nr].s, val);
		}
	}
}

void
sig_ide_bus_irq(struct sig_ide_bus *b, void *s, unsigned int val)
{
	unsigned int oldresval;
	unsigned int newresval;
	unsigned int nr;

	assert(/* 0 <= val && */ val <= 1);

	oldresval = 0;
	newresval = 0;

	/*
	 * Compute old and new b value.
	 */
	for (nr = 0; ; nr++) {
		if (nr == b->nmembers) {
			break;
		}
		oldresval |= b->member[nr].irq_state;
		if (b->member[nr].s == s) {
			b->member[nr].irq_state = val;
		}
		newresval |= b->member[nr].irq_state;
	}

	/*
	 * If old value equal new value do nothing.
	 */
	if (oldresval == newresval) {
		return;
	}

	/*
	 * Call callback function.
	 */
	for (nr = 0; ; nr++) {
		void (*func)(void *, unsigned int);

		if (nr == b->nmembers) {
			break;
		}
		func = b->member[nr].funcs->irq;
		if (func) {
			func(b->member[nr].s, newresval);
		}
	}
}

void
sig_ide_bus_dmarq_set(struct sig_ide_bus *b, unsigned int val)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		void (*func)(void *, unsigned int);

		if (nr == b->nmembers) {
			return;
		}
		func = b->member[nr].funcs->dmarq_set;
		if (func) {
			func(b->member[nr].s, val);
			return;
		}
	}
}

void
sig_ide_bus_connect(
	struct sig_ide_bus *b,
	void *s,
	const struct sig_ide_bus_funcs *funcs
)
{
	assert(b);
	assert(b->type == SIG_GEN_IDE_BUS);
	assert(b->nmembers < 3);

	b->member[b->nmembers].s = s;
	b->member[b->nmembers].funcs = funcs;
	b->member[b->nmembers].irq_state = 0;
	b->nmembers++;
}

struct sig_ide_bus_merge *
sig_ide_bus_merge(struct sig_ide_bus *s0, struct sig_ide_bus *s1)
{
	fixme();
	return NULL;
}

void
sig_ide_bus_split(struct sig_ide_bus_merge *m)
{
	fixme();
}

struct sig_ide_bus *
sig_ide_bus_create(const char *name)
{
	struct sig_ide_bus *b;

	b = shm_alloc(sizeof(struct sig_ide_bus));
	assert(b);

	b->type = SIG_GEN_IDE_BUS;
	b->nmembers = 0;

	return b;
}

void
sig_ide_bus_destroy(struct sig_ide_bus *b)
{
	assert(b);
	assert(b->type == SIG_GEN_IDE_BUS);
	
	shm_free(b);
}

void
sig_ide_bus_suspend(struct sig_ide_bus *b, FILE *fp)
{
	size_t size = sizeof(*b);
	
	generic_suspend(b, size, fp);
}

void
sig_ide_bus_resume(struct sig_ide_bus *b, FILE *fp)
{
	size_t size = sizeof(*b);
	
	generic_resume(b, size, fp);
}
