/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_CONTROL_FLOW	1

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#define INCLUDE
#include "arch_i2c_slave.c"
#undef INCLUDE

#include "glue.h"

#include "chip_st_lsm303dlhc.h"

#define CHIP_(x) chip_st_lsm303dlhc_ ## x

struct cpssp {
	unsigned int state_power;

	unsigned int acc_sda_state;
	unsigned int mag_sda_state;
	struct sig_std_logic *port_sda;

	int state_acc_x;
	int state_acc_y;
	int state_acc_z;
	int state_mag_x;
	int state_mag_y;
	int state_mag_z;

	/* ACC */
	int acc_is_address_phase;
	uint8_t acc_counter;

	uint8_t acc_ctrl_reg1;
	uint8_t acc_ctrl_reg2;
	uint8_t acc_ctrl_reg3;

	/* CTRL_REG4_A */
	uint8_t acc_bdu;
	uint8_t acc_ble;
	uint8_t acc_fs;
	uint8_t acc_hr;
	uint8_t acc_sim;

	uint8_t acc_ctrl_reg5;
	uint8_t acc_ctrl_reg6;
	uint8_t acc_reference;
	uint8_t acc_status_reg;
	uint8_t acc_fifo_ctrl_reg;
	uint8_t acc_fifo_src_reg;
	uint8_t acc_int1_cfg;
	uint8_t acc_int1_source;
	uint8_t acc_int1_ths;
	uint8_t acc_int1_duration;
	uint8_t acc_int2_cfg;
	uint8_t acc_int2_source;
	uint8_t acc_int2_ths;
	uint8_t acc_int2_duration;
	uint8_t acc_click_cfg;
	uint8_t acc_click_src;
	uint8_t acc_click_ths;
	uint8_t acc_time_limit;
	uint8_t acc_time_latency;
	uint8_t acc_time_window;

	/* MAG */
	int mag_is_address_phase;
	uint8_t mag_counter;

#define STATE

#define NAME		i2c_acc
#define NAME_(x)	i2c_acc_ ## x
#include "arch_i2c_slave.c"
#undef NAME_
#undef NAME

#define NAME		i2c_mag
#define NAME_(x)	i2c_mag_ ## x
#include "arch_i2c_slave.c"
#undef NAME_
#undef NAME

#undef STATE
};

/*
 * ACC
 */
static void
CHIP_(acc_inc)(struct cpssp *cpssp)
{
	if ((cpssp->acc_counter >> 7) & 1) {
		cpssp->acc_counter = 0x80 | ((cpssp->acc_counter + 1) & 0x7f);
	}
}

static void
CHIP_(acc_reg_write)(struct cpssp *cpssp, uint8_t addr, uint8_t val)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%02x val=0x%02x\n",
				__FUNCTION__, addr, val);
	}

	switch (addr) {
	case 0x0f:
		/* Who am I - not documented. */
	/* read-only: */
		fprintf(stderr, "WARNING: %s: addr=0x%02x val=0x%02x\n",
				__FUNCTION__, addr, val);
		break;
	case 0x23:
		/* CTRL_REG4_A */
		cpssp->acc_bdu = (val >> 7) & 1;
		cpssp->acc_ble = (val >> 6) & 1;
		cpssp->acc_fs = (val >> 4) & 3;
		cpssp->acc_hr = (val >> 3) & 1;
		cpssp->acc_sim = (val >> 0) & 1;
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%02x val=0x%02x\n",
				__FUNCTION__, addr, val);
		break;
	}
}

static uint8_t
CHIP_(acc_reg_read)(struct cpssp *cpssp, uint8_t addr)
{
	int value;
	uint8_t val;

	switch (addr) {
	case 0x00 ... 0x0e:
		/* Reserved */
		goto reserved;
	case 0x0f:
		/* Who am I - Not documented. */
		val = 0x33;
		break;
	case 0x10 ... 0x1f:
		/* Reserved */
		goto reserved;
	case 0x20:
		val = cpssp->acc_ctrl_reg1;
		break;
	case 0x21:
		val = cpssp->acc_ctrl_reg2;
		break;
	case 0x22:
		val = cpssp->acc_ctrl_reg3;
		break;
	case 0x23:
		/* CTRL_REG4_A */
		val = 0;
		val |= cpssp->acc_bdu << 7;
		val |= cpssp->acc_ble << 6;
		val |= cpssp->acc_fs << 4;
		val |= cpssp->acc_hr << 3;
		val |= cpssp->acc_sim << 0;
		break;
	case 0x24:
		val = cpssp->acc_ctrl_reg5;
		break;
	case 0x25:
		val = cpssp->acc_ctrl_reg6;
		break;
	case 0x26:
		val = cpssp->acc_reference;
		break;
	case 0x27:
		val = cpssp->acc_status_reg;
		break;
	case 0x28:
		/* OUT_X_L_A */
	case 0x29:
		/* OUT_X_H_A */
		value = (cpssp->state_acc_x << 15) / 1024;
		assert((int)(int16_t) value == value);

		if (cpssp->acc_ble ^ (addr & 1)) {
			/* Big Endian XOR High-Byte read */
			val = (value >> 8) & 0xff;
		} else {
			val = (value >> 0) & 0xff;
		}
		break;
	case 0x2a:
		/* OUT_Y_L_A */
	case 0x2b:
		/* OUT_Y_H_A */
		value = (cpssp->state_acc_y << 15) / 1024;
		assert((int)(int16_t) value == value);

		if (cpssp->acc_ble ^ (addr & 1)) {
			/* Big Endian XOR High-Byte read */
			val = (value >> 8) & 0xff;
		} else {
			val = (value >> 0) & 0xff;
		}
		break;
	case 0x2c:
		/* OUT_Z_L_A */
	case 0x2d:
		/* OUT_Z_H_A */
		value = (cpssp->state_acc_z << 15) / 1024;
		assert((int)(int16_t) value == value);

		if (cpssp->acc_ble ^ (addr & 1)) {
			/* Big Endian XOR High-Byte read */
			val = (value >> 8) & 0xff;
		} else {
			val = (value >> 0) & 0xff;
		}
		break;
	case 0x2e:
		val = cpssp->acc_fifo_ctrl_reg;
		break;
	case 0x2f:
		val = cpssp->acc_fifo_src_reg;
		break;
	case 0x30:
		val = cpssp->acc_int1_cfg;
		break;
	case 0x31:
		val = cpssp->acc_int1_source;
		break;
	case 0x32:
		val = cpssp->acc_int1_ths;
		break;
	case 0x33:
		val = cpssp->acc_int1_duration;
		break;
	case 0x34:
		val = cpssp->acc_int2_cfg;
		break;
	case 0x35:
		val = cpssp->acc_int2_source;
		break;
	case 0x36:
		val = cpssp->acc_int2_ths;
		break;
	case 0x37:
		val = cpssp->acc_int2_duration;
		break;
	case 0x38:
		val = cpssp->acc_click_cfg;
		break;
	case 0x39:
		val = cpssp->acc_click_src;
		break;
	case 0x3a:
		val = cpssp->acc_click_ths;
		break;
	case 0x3b:
		val = cpssp->acc_time_limit;
		break;
	case 0x3c:
		val = cpssp->acc_time_latency;
		break;
	case 0x3d:
		val = cpssp->acc_time_window;
		break;
	case 0x3e ... 0x3f:
	case 0x40 ... 0x7f:
	reserved:
		fprintf(stderr, "WARNING: %s: addr=0x%02x\n",
				__FUNCTION__, addr);
		val = 0x00;
		break;
	default:
		assert(0); /* Mustn't happen. */
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%02x val=0x%02x\n",
				__FUNCTION__, addr, val);
	}

	return val;
}

static void
CHIP_(acc_reset)(struct cpssp *cpssp)
{
	cpssp->acc_ctrl_reg1 = 0b00000111;
	cpssp->acc_ctrl_reg2 = 0;
	cpssp->acc_ctrl_reg3 = 0;

	/* CTRL_REG4_A */
	cpssp->acc_bdu = 0;
	cpssp->acc_ble = 0;
	cpssp->acc_hr = 0;
	cpssp->acc_fs = 0;
	cpssp->acc_sim = 0;

	cpssp->acc_ctrl_reg5 = 0;
	cpssp->acc_ctrl_reg6 = 0;
	cpssp->acc_reference = 0;
	cpssp->acc_status_reg = 0;
	cpssp->acc_fifo_ctrl_reg = 0;
	cpssp->acc_fifo_src_reg = 0;
	cpssp->acc_int1_cfg = 0;
	cpssp->acc_int1_source = 0;
	cpssp->acc_int1_ths = 0;
	cpssp->acc_int1_duration = 0;
	cpssp->acc_int2_cfg = 0;
	cpssp->acc_int2_source = 0;
	cpssp->acc_int2_ths = 0;
	cpssp->acc_int2_duration = 0;
	cpssp->acc_click_cfg = 0;
	cpssp->acc_click_src = 0;
	cpssp->acc_click_ths = 0;
	cpssp->acc_time_limit = 0;
	cpssp->acc_time_latency = 0;
	cpssp->acc_time_window = 0;
}

/*
 * MAG
 */
static void
CHIP_(mag_inc)(struct cpssp *cpssp)
{
	if (cpssp->mag_counter == 8) {
		cpssp->mag_counter = 3;
	} else if (12 <= cpssp->mag_counter) {
		cpssp->mag_counter = 0;
	} else {
		cpssp->mag_counter++;
	}
}

static void
CHIP_(mag_reg_write)(struct cpssp *cpssp, uint8_t addr, uint8_t val)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%02x val=0x%02x\n",
				__FUNCTION__, addr, val);
	}

	switch (addr) {
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%02x val=0x%02x\n",
				__FUNCTION__, addr, val);
		break;
	}
}

static uint8_t
CHIP_(mag_reg_read)(struct cpssp *cpssp, uint8_t addr)
{
	uint8_t val;

	switch (addr) {
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%02x\n",
				__FUNCTION__, addr);
		val = 0x00;
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%02x val=0x%02x\n",
				__FUNCTION__, addr, val);
	}

	return val;
}

static void
CHIP_(mag_reset)(struct cpssp *cpssp)
{
}

/*
 * Callbacks for ACC interface.
 */
static int
i2c_acc_ack_addr(struct cpssp *cpssp, uint8_t val)
{
	return (val & 0xfe) == 0x32;
}

static void
i2c_acc_read_byte(struct cpssp *cpssp, uint8_t *valp)
{
	*valp = CHIP_(acc_reg_read)(cpssp, cpssp->acc_counter & 0x7f);
	CHIP_(acc_inc)(cpssp);
}

static void
i2c_acc_write_byte(struct cpssp *cpssp, uint8_t val)
{
	if (cpssp->acc_is_address_phase) {
		cpssp->acc_counter = val;
		cpssp->acc_is_address_phase = 0;

	} else {
		CHIP_(acc_reg_write)(cpssp, cpssp->acc_counter & 0x7f, val);
		CHIP_(acc_inc)(cpssp);
	}
}

static void
i2c_acc_stop_transaction(struct cpssp *cpssp)
{
	cpssp->acc_is_address_phase = 1;
}

static void
i2c_acc_sda_out(struct cpssp *cpssp, unsigned int val)
{
	cpssp->acc_sda_state = val;

	val = sig_std_logic_resolve(cpssp->acc_sda_state, cpssp->mag_sda_state);

	sig_std_logic_set(cpssp->port_sda, cpssp, val);
}

/*
 * Callbacks for MAG interface.
 */
static int
i2c_mag_ack_addr(struct cpssp *cpssp, uint8_t val)
{
	return (val & 0xfe) == 0x3c;
}

static void
i2c_mag_read_byte(struct cpssp *cpssp, uint8_t *valp)
{
	*valp = CHIP_(mag_reg_read)(cpssp, cpssp->mag_counter & 0x7f);
	CHIP_(mag_inc)(cpssp);
}

static void
i2c_mag_write_byte(struct cpssp *cpssp, uint8_t val)
{
	if (cpssp->mag_is_address_phase) {
		cpssp->mag_counter = val;
		cpssp->mag_is_address_phase = 0;

	} else {
		CHIP_(mag_reg_write)(cpssp, cpssp->mag_counter, val);
		CHIP_(mag_inc)(cpssp);
	}
}

static void
i2c_mag_stop_transaction(struct cpssp *cpssp)
{
	cpssp->mag_is_address_phase = 1;
}

static void
i2c_mag_sda_out(struct cpssp *cpssp, unsigned int val)
{
	cpssp->mag_sda_state = val;

	val = sig_std_logic_resolve(cpssp->acc_sda_state, cpssp->mag_sda_state);

	sig_std_logic_set(cpssp->port_sda, cpssp, val);
}

#define BEHAVIOR

#define NAME		i2c_acc
#define NAME_(x)	i2c_acc_ ## x
#include "arch_i2c_slave.c"
#undef NAME_
#undef NAME

#define NAME		i2c_mag
#define NAME_(x)	i2c_mag_ ## x
#include "arch_i2c_slave.c"
#undef NAME_
#undef NAME

#undef BEHAVIOR

static void
CHIP_(reset)(struct cpssp *cpssp)
{
	cpssp->acc_is_address_phase = 1;
	cpssp->mag_is_address_phase = 1;

	CHIP_(acc_reset)(cpssp);
	CHIP_(mag_reset)(cpssp);
}

static void
CHIP_(scl_in_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	i2c_acc_scl_in(cpssp, val);
	i2c_mag_scl_in(cpssp, val);
}

static void
CHIP_(sda_in_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	i2c_acc_sda_in(cpssp, val);
	i2c_mag_sda_in(cpssp, val);
}

static void
CHIP_(vdd_in_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_power = val;
	if (val) {
		/* Power-on reset. */
		CHIP_(reset)(cpssp);
	}
}

static void
CHIP_(acc_x_in_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(-1024 < val && val < 1024);

	cpssp->state_acc_x = val;
}

static void
CHIP_(acc_y_in_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(-1024 < val && val < 1024);

	cpssp->state_acc_y = val;
}

static void
CHIP_(acc_z_in_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(-1024 < val && val < 1024);

	cpssp->state_acc_z = val;
}

static void
CHIP_(mag_x_in_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(-1024 < val && val < 1024);

	cpssp->state_mag_x = val;
}

static void
CHIP_(mag_y_in_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(-1024 < val && val < 1024);

	cpssp->state_mag_y = val;
}

static void
CHIP_(mag_z_in_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(-1024 < val && val < 1024);

	cpssp->state_mag_z = val;
}

void *
CHIP_(create)(
	const char *name,
	struct sig_manage *manage,
	struct sig_std_logic *port_gnd,
	struct sig_std_logic *port_vdd,
	struct sig_std_logic *port_vdd_io,
	struct sig_std_logic *port_scl,
	struct sig_std_logic *port_sda,
	struct sig_std_logic *port_int1,
	struct sig_std_logic *port_int2,
	struct sig_std_logic *port_c1,
	struct sig_std_logic *port_drdy,
	struct sig_std_logic *port_setp,
	struct sig_std_logic *port_setc,
	struct sig_std_logic *port_reserved1,
	struct sig_std_logic *port_reserved2,
	struct sig_std_logic *port_reserved3,
	struct sig_integer *port_acc_x,
	struct sig_integer *port_acc_y,
	struct sig_integer *port_acc_z,
	struct sig_integer *port_mag_x,
	struct sig_integer *port_mag_y,
	struct sig_integer *port_mag_z
)
{
	static const struct sig_std_logic_funcs vdd_funcs = {
		.boolean_or_set = CHIP_(vdd_in_set),
	};
	static const struct sig_std_logic_funcs scl_funcs = {
		.std_logic_set = CHIP_(scl_in_set),
	};
	static const struct sig_std_logic_funcs sda_funcs = {
		.std_logic_set = CHIP_(sda_in_set),
	};
	static const struct sig_integer_funcs acc_x_funcs = {
		.set = CHIP_(acc_x_in_set),
	};
	static const struct sig_integer_funcs acc_y_funcs = {
		.set = CHIP_(acc_y_in_set),
	};
	static const struct sig_integer_funcs acc_z_funcs = {
		.set = CHIP_(acc_z_in_set),
	};
	static const struct sig_integer_funcs mag_x_funcs = {
		.set = CHIP_(mag_x_in_set),
	};
	static const struct sig_integer_funcs mag_y_funcs = {
		.set = CHIP_(mag_y_in_set),
	};
	static const struct sig_integer_funcs mag_z_funcs = {
		.set = CHIP_(mag_z_in_set),
	};
	struct cpssp *cpssp;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->acc_sda_state = SIG_STD_LOGIC_Z;
	cpssp->mag_sda_state = SIG_STD_LOGIC_Z;

	/* FIXME */
	i2c_acc_create(cpssp);
	i2c_mag_create(cpssp);

	CHIP_(reset)(cpssp); /* FIXME */

	/* Out */
	cpssp->port_sda = port_sda;
	sig_std_logic_connect_out(port_sda, cpssp, SIG_STD_LOGIC_Z);

	/* In */
	cpssp->state_power = 0;
	sig_std_logic_connect_in(port_vdd, cpssp, &vdd_funcs);
	sig_std_logic_connect_in(port_scl, cpssp, &scl_funcs);
	sig_std_logic_connect_in(port_sda, cpssp, &sda_funcs);

	cpssp->state_acc_x = 0;
	sig_integer_connect_in(port_acc_x, cpssp, &acc_x_funcs);
	cpssp->state_acc_y = 0;
	sig_integer_connect_in(port_acc_y, cpssp, &acc_y_funcs);
	cpssp->state_acc_z = 0;
	sig_integer_connect_in(port_acc_z, cpssp, &acc_z_funcs);
	cpssp->state_mag_x = 0;
	sig_integer_connect_in(port_mag_x, cpssp, &mag_x_funcs);
	cpssp->state_mag_y = 0;
	sig_integer_connect_in(port_mag_y, cpssp, &mag_y_funcs);
	cpssp->state_mag_z = 0;
	sig_integer_connect_in(port_mag_z, cpssp, &mag_z_funcs);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	/* FIXME */
	i2c_acc_destroy(cpssp);
	i2c_mag_destroy(cpssp);

	free(cpssp);
}

void
CHIP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}

#undef DEBUG_CONTROL_FLOW
