/*	main.c
	Copyright (C) 2004-2007 Mark Tyler

	This file is part of rgbPaint.

	rgbPaint is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	rgbPaint is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with rgbPaint in the file COPYING.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <fcntl.h>
#include <signal.h>

#include "global.h"
#include "memory.h"
#include "png.h"
#include "mainwindow.h"
#include "otherwindow.h"
#include "viewer.h"
#include "canvas.h"
#include "toolbar.h"

static int olpc_widget;
int thumb_size = 40;

#ifndef WIN32

#include <unistd.h>

static int fh;


static void pipe_signals(int signal)
{
	if(write(fh, &signal, sizeof(int)) != sizeof(int))
	{
		fprintf(stderr, "unix signal %d lost\n", signal);
	}
}

static gboolean deliver_signal(GIOChannel *source, GIOCondition cond, gpointer d)
{
	GError *error = NULL;
	union {
		gchar chars[128];
		int signal;
		} buf;
	GIOStatus status;
	gsize bytes_read;
	char *res;
	int r, g, b;


	/* 
	 * Read from the pipe as long as data is available. The reading end is 
	 * also in non-blocking mode, so if we have consumed all unix signals, 
	 * the read returns G_IO_STATUS_AGAIN. 
	 */
	while((status = g_io_channel_read_chars(source, buf.chars, 
		128, &bytes_read, &error)) == G_IO_STATUS_NORMAL)
	{
		g_assert(error == NULL);	// no error if reading returns normal

		res = strstr(buf.chars, "ACTIVATE");
		if (res)
		{
			// Activate message received from Sugar
		}
		res = strstr(buf.chars, "QUIT");
		if (res)
		{
			delete_event( NULL, NULL, NULL);	// QUIT message received from Sugar
		}
		res = strstr(buf.chars, "COLOUR");
		if (res)
		{
				// Sugar wants to change the current colour
			sscanf(buf.chars, "COLOUR %i %i %i", &r, &g, &b);

			mem_pal[mem_col_A].red   = r;
			mem_pal[mem_col_A].green = g;
			mem_pal[mem_col_A].blue  = b;

			mem_col_A24 = mem_pal[mem_col_A];
			mem_col_B24 = mem_pal[mem_col_B];
			init_pal();
			toolbar_palette_refresh();
		}
	}

	g_assert(status == G_IO_STATUS_AGAIN);
	return (TRUE);				// keep the event source
}
#endif



int main( int argc, char *argv[] )
{
#ifndef WIN32
	GIOChannel *g_signal_in;
	long fd_flags;			// used to change the pipe into non-blocking mode
	GError *error = NULL;		// handle errors
#endif

	char ppath[256];
	int i;
	gboolean new_empty = TRUE, get_screenshot = FALSE;

	global_argc = argc;

	if (argc > 1)
	{
		if ( strcmp(argv[1], "--version") == 0 )
		{
			printf("%s\n\n", VERSION);
			exit(0);
		}
		if ( strcmp(argv[1], "--help") == 0 )
		{
			printf("%s\n\n"
				"Usage: rgbpaint [options] [imagefile]\n\n"
				"Options:\n"
				"  --help          Output this help\n"
				"  --version       Output version information\n"
				"  -d <dir>        Set default directory for load/save\n"
				"  -s              Grab screenshot\n"
				"  -u <limit>      Set undo limit in megabytes, 1..500\n"
				"  -stamps <...>   All remaining files passed are loaded as stamps\n"
				"  -thumb <size>   Set the maximum width/height of the stamp thumbnails 32..256\n"
				"  -svg <dir>      Load SVG icons from directory\n"
				"\n"
			, VERSION);
			exit(0);
		}
	}

	global_argv = argv;
	putenv( "G_BROKEN_FILENAMES=1" );	// Needed to read non ASCII filenames in GTK+2

	gtk_init( &argc, &argv );


#ifdef U_NLS
	gtk_set_locale();
	bindtextdomain("rgbpaint", MT_LANG_DEST);
	textdomain("rgbpaint");
	bind_textdomain_codeset("rgbpaint", "UTF-8");
#endif


	files_passed = argc - 1;
	if (argc > 1)		// Arguments received
	{
		file_arg_start = 1;

		for ( i=1; i<argc; i++ )
		{
			if ( strcmp(argv[i], "-svg") == 0 )	// Set SVG directory
			{
				file_arg_start+=2;
				files_passed-=2;
				i++;
				svg_dir = argv[i];
			}
			if ( strcmp(argv[i], "-thumb") == 0 )	// Set thumbnail size
			{
				file_arg_start+=2;
				files_passed-=2;
				i++;
				sscanf( argv[i], "%i", &thumb_size );
				if ( thumb_size<32 || thumb_size>256 )
					thumb_size = 40;	// Ensure sanity
			}
			if ( strcmp(argv[i], "-stamps") == 0 )	// Files are stamps
			{
				file_arg_start++;
				files_passed--;
				stamp_start = i+1;
			}
			if ( strcmp(argv[i], "-d") == 0 )	// Set the default directory
			{
				file_arg_start += 2;
				files_passed -= 2;
				i++;
				fs_set_dir(argv[i]);
			}
			if ( strcmp(argv[i], "-s") == 0 )	// Screenshot
			{
				file_arg_start++;
				files_passed--;
				get_screenshot = TRUE;
			}
			if ( strcmp(argv[i], "-u") == 0 )	// Undo memory limit
			{
				file_arg_start+=2;
				files_passed-=2;
				i++;
				sscanf( argv[i], "%i", &mem_undo_limit );
				if ( mem_undo_limit<1 || mem_undo_limit>500 )
					mem_undo_limit = 32;	// Ensure sanity
			}
#ifndef WIN32
			if ( strcmp(argv[i], "-browserWindow") == 0 )
			{
				file_arg_start+=2;
				files_passed-=2;
				i++;
				sscanf( argv[i], "%i", &olpc_widget );
			}
			if ( strcmp(argv[i], "sugarPipe") == 0 )
			{
				file_arg_start+=2;
				files_passed-=2;
				i++;
				fh = open(argv[i], O_RDWR);
						// Open socket file to grab messages from Sugar

				fd_flags = fcntl(fh, F_GETFL);
				if(fd_flags == -1)
				{
					// error - read descriptor flags
					perror("write descriptor flags");
					exit(1);
				}
				if(fcntl(fh, F_SETFL, fd_flags | O_NONBLOCK) == -1)
				{
					perror("write descriptor flags");
					exit(1);
				}

				/* Install the unix signal handler pipe_signals for the signals of interest */
				signal(SIGINT, pipe_signals);
				signal(SIGUSR1, pipe_signals);

				/* convert the reading end of the pipe into a GIOChannel */
				g_signal_in = g_io_channel_unix_new(fh);

				/* 
				 * we only read raw binary data from the pipe, 
				 * therefore clear any encoding on the channel
				 */
				g_io_channel_set_encoding(g_signal_in, NULL, &error);
				if(error != NULL)
				{		// handle potential errors
					fprintf(stderr, "g_io_channel_set_encoding failed %s\n",
							error->message);
					exit(1);
				}

				/* put the reading end also into non-blocking mode */
				g_io_channel_set_flags(g_signal_in,
					g_io_channel_get_flags(g_signal_in) | G_IO_FLAG_NONBLOCK,
					&error);

				if(error != NULL)
				{			// tread errors
					fprintf(stderr, "g_io_set_flags failed %s\n",
						error->message);
					exit(1);
				}

				/* register the reading end with the event loop */
				g_io_add_watch(g_signal_in, G_IO_IN | G_IO_PRI, deliver_signal, NULL);
			}
#endif
		}
	}

	mem_init();					// Set up memory & back end

	if ( get_screenshot )
	{
		if ( grab_screen() )
			new_empty = FALSE;		// Successfully grabbed so no new empty
		else
			get_screenshot = FALSE;		// Screenshot failed
	}
	main_init(olpc_widget);					// Create main window

	if ( get_screenshot )
	{
		do_new_chores();
		notify_changed();
	}
	else
	{
		if (files_passed >0 && stamp_start==-1)
		{
			strncpy( ppath, argv[file_arg_start], 250 );
			if ( do_a_load( ppath ) == 0 )
			{
				new_empty = FALSE;
			}
		}
	}

	if ( new_empty )		// If no file was loaded, start with a blank canvas
	{
		create_default_image();
	}
	update_menus();

	gtk_main();

	return 0;
}
